package models.consum;

import java.text.ParseException;
import java.text.SimpleDateFormat;
import java.util.ArrayList;
import java.util.Date;
import java.util.List;

import javax.persistence.CascadeType;
import javax.persistence.Column;
import javax.persistence.Entity;
import javax.persistence.GeneratedValue;
import javax.persistence.GenerationType;
import javax.persistence.Id;
import javax.persistence.OneToMany;
import javax.persistence.OrderBy;
import javax.persistence.Table;
import javax.persistence.Transient;

import com.avaje.ebean.Model;
import com.fasterxml.jackson.databind.node.ObjectNode;

import models.contactix.Address;
import models.contactix.PhoneNumber;
import models.contactix.User;
import play.Logger;
import play.libs.Json;

/**
 * Created by philipp on 12.03.14.
 */
@Entity
@Table(name = "clients")
public class Client extends Model {
	@Id
	@GeneratedValue(strategy = GenerationType.IDENTITY)
	public Integer id;
	@Column(name = "client_no", columnDefinition = "int(11)")
	public Integer clientNo;
	@Column(name = "billing_interval", columnDefinition = "int(11)")
	public Integer billingInterval;
	@Column(name = "user_id", columnDefinition = "int(11)")
	public Integer userId;

	public String name;
	@Column(length = 40)
	public String company;
	@Column(name = "email_address")
	public String email;
	public String locale;
	@Column(name = "workflow_state")
	public String workflowState;
	@Column(name = "created_at")
	public Date createdAt = new Date();
	@Column(name = "updated_at")
	public Date updatedAt = new Date();
	@Column(name = "deleted_at")
	public Date deletedAt = null;
	
	@Transient
	private User user = null;
	@Transient
	private Address preferedAddress;
	@Transient
	private PhoneNumber preferedPhoneNumber;
	
	private final static SimpleDateFormat DATETIME_FORMAT = new SimpleDateFormat("dd.MM.yyyy HH:mm:ss");

	public static Finder<Integer, Client> find = new Finder<Integer, Client>("consum", Integer.class, Client.class);
	
	/**
	 * Get the Matching User for this Client.
	 * 
	 * @return
	 */
	public User getUser() {
		if (user == null) {
			user = User.find.where().where().eq("id", this.userId).findUnique();
		}
		return user;
	}
	
	public String getUsername() {
		if (getUser() != null) {
			return user.name;
		}		
		return null;
	}

	/**
	 * Get a List of Users, this Client is responsible for.
	 * 
	 * @return
	 */
	public List<User> getUsers() {
		return User.find.where().where().eq("responsibleClientId", this.id)
				.findList();
	}

	public List<Address> getAddresses() {
		if (getUser() != null) {
			return user.getAddresses();
		}
		return new ArrayList<Address>();
	}

	public Address getPreferedAddress() {
		if (preferedAddress == null && getUser() != null) {
			preferedAddress = user.getPreferedAddress();
		}
		return preferedAddress;		
	}
	
	public void setPreferedAddress(final Address preferedAddress){
		this.preferedAddress = preferedAddress;
	}
	
	public List<PhoneNumber> getPhoneNumbers() {
		if (getUser() != null) {
			return user.phoneNumbers;
		}
		return new ArrayList<PhoneNumber>();
	}
	
	public PhoneNumber getPreferedPhoneNumber() {
		if (preferedPhoneNumber == null && getUser() != null) {
			preferedPhoneNumber = user.getPreferedPhoneNumber();
		}
		return preferedPhoneNumber;		
	}
	
	public void setPreferedPhoneNumber(final PhoneNumber preferedPhoneNumber) {
		this.preferedPhoneNumber = preferedPhoneNumber;
	}
	
	public String toString() {
		return name;
	}
	
	public ObjectNode toJson() {
		ObjectNode result = Json.newObject();
		result.put("id", id);
		result.put("createdAt", DATETIME_FORMAT.format(createdAt));
		result.put("updatedAt", DATETIME_FORMAT.format(updatedAt));
		if(user != null) {
			result.set("user", user.toJson());
		}
		result.put("clientNo", clientNo);
		result.put("billingInterval", billingInterval);
		result.put("name", name);
		result.put("company", company);
		result.put("email", email);
		result.put("locale", locale);
		result.put("workflowState", workflowState);
		Address preferedAddress = getPreferedAddress();
		result.set("preferedAddress", preferedAddress != null ? preferedAddress.toJson() : null);
		PhoneNumber preferedPhoneNumber = getPreferedPhoneNumber();
		result.set("preferedPhoneNumber", preferedPhoneNumber != null ? preferedPhoneNumber.toJson() : null);
		return result;
	}
	
	public static Client fromJson(final ObjectNode json) {
		Client client = new Client();
		client.id = json.get("id").asInt();
		try {
			client.createdAt =  DATETIME_FORMAT.parse(json.get("createdAt").asText());
			client.updatedAt =  DATETIME_FORMAT.parse(json.get("updatedAt").asText());
			// TODO add additional attributes
		} catch (ParseException ex) {
			Logger.error(ex.getLocalizedMessage(), ex);
		}
		return client;
	}
	
	public static Client getUserClient(User user) {
		return Client.find.where().eq("user_id", user.id).findUnique();
	}
		
}
