/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at legal-notices/CDDLv1_0.txt
 * or http://forgerock.org/license/CDDLv1.0.html.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at legal-notices/CDDLv1_0.txt.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2006-2008 Sun Microsystems, Inc.
 *      Portions Copyright 2012-2014 ForgeRock AS.
 */
package org.opends.dsml.protocol;



import java.io.IOException;
import java.util.List;

import org.forgerock.i18n.LocalizableMessage;
import org.forgerock.opendj.ldap.DecodeException;
import org.opends.server.protocols.ldap.DeleteRequestProtocolOp;
import org.opends.server.protocols.ldap.DeleteResponseProtocolOp;
import org.opends.server.protocols.ldap.LDAPMessage;
import org.opends.server.protocols.ldap.ProtocolOp;
import org.opends.server.tools.LDAPConnection;
import org.forgerock.opendj.ldap.ByteString;
import org.opends.server.types.DN;
import org.opends.server.types.LDAPException;


/**
 * This class provides the functionality for the performing an
 * LDAP DELETE operation based on the specified DSML request.
 *
 *
 * @author   Vivek Nagar
 */


public class DSMLDeleteOperation
{
  private LDAPConnection connection;

  /**
   * Create an instance with the specified LDAP connection.
   *
   * @param connection    The LDAP connection to send the request on.
   */
  public DSMLDeleteOperation(LDAPConnection connection)
  {
    this.connection = connection;
  }

  /**
   * Perform the LDAP DELETE operation and send the result back to the
   * client.
   *
   * @param  objFactory     The object factory for this operation.
   * @param  deleteRequest  The delete request for this operation.
   * @param  controls       Any required controls (e.g. for proxy authz).
   *
   * @return  The result of the delete operation.
   *
   * @throws  IOException  If an I/O problem occurs.
   *
   * @throws  LDAPException  If an error occurs while interacting with an LDAP
   *                         element.
   *
   * @throws  DecodeException  If an error occurs while interacting with an ASN.1
   *                         element.
   */
  public LDAPResult doOperation(ObjectFactory objFactory,
        DelRequest deleteRequest,
        List<org.opends.server.types.Control> controls)
    throws IOException, LDAPException, DecodeException
  {
    LDAPResult delResponse = objFactory.createLDAPResult();
    delResponse.setRequestID(deleteRequest.getRequestID());

    // Create and send the LDAP delete request to the server.
    ByteString dnStr = ByteString.valueOf(deleteRequest.getDn());
    ProtocolOp op = new DeleteRequestProtocolOp(dnStr);
    LDAPMessage msg = new LDAPMessage(DSMLServlet.nextMessageID(), op,
        controls);
    connection.getLDAPWriter().writeMessage(msg);

    // Read and decode the LDAP response from the server.
    LDAPMessage responseMessage = connection.getLDAPReader().readMessage();

    DeleteResponseProtocolOp delOp =
          responseMessage.getDeleteResponseProtocolOp();
    int resultCode = delOp.getResultCode();
    LocalizableMessage errorMessage = delOp.getErrorMessage();

    // Set the result code and error message for the DSML response.
    delResponse.setErrorMessage(
            errorMessage != null ? errorMessage.toString() : null);
    ResultCode code = ResultCodeFactory.create(objFactory, resultCode);
    delResponse.setResultCode(code);

    // set the match DN
    DN dn = delOp.getMatchedDN();
    if ( dn != null ) {
      delResponse.setMatchedDN(dn.toString());
    }

    return delResponse;
  }

}

