/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at legal-notices/CDDLv1_0.txt
 * or http://forgerock.org/license/CDDLv1.0.html.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at legal-notices/CDDLv1_0.txt.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2008 Sun Microsystems, Inc.
 *      Portions Copyright 2013-2015 ForgeRock AS
 */
package org.opends.server.authorization.dseecompat;
import org.forgerock.i18n.LocalizableMessage;

import static org.opends.messages.AccessControlMessages.*;
import org.opends.server.util.TimeThread;
import java.util.regex.Pattern;

/**
 * This class represents the timeofday keyword in a bind rule.
 */
public class TimeOfDay implements KeywordBindRule {

    /**
     * Regular expression matching a valid timeofday rule value (0-2359).
     */
    private static final String timeofdayRegex = "[0-2]\\d[0-5]\\d";

    /** Enumeration representing the bind rule operation type. */
    private EnumBindRuleType type;

    /** Holds the time value parsed from the ACI. */
    private int timeRef;

    /**
     * Constructor to create a timeofday keyword class.
     * @param timeVal The time value to check for (0-2359).
     * @param type An enumeration of the type of the expression.
     */
    private TimeOfDay(int timeVal, EnumBindRuleType type) {
        this.timeRef=timeVal;
        this.type=type;
    }

    /**
     * Decodes a string representation of a timeofday bind rule expression.
     * @param expr A string representation of the expression.
     * @param type An enumeration of the type of the expression.
     * @return  A TimeOfDay class representing the expression.
     * @throws AciException If the expression is invalid.
     */
    public static TimeOfDay decode(String expr,  EnumBindRuleType type)
    throws AciException  {
        int valueAsInt = 0;
        if (!Pattern.matches(timeofdayRegex, expr))
        {
            LocalizableMessage message = WARN_ACI_SYNTAX_INVALID_TIMEOFDAY.get(expr);
            throw new AciException(message);
         }
        try {
            valueAsInt = Integer.parseInt(expr);
        } catch (NumberFormatException nfe) {
          LocalizableMessage message =
           WARN_ACI_SYNTAX_INVALID_TIMEOFDAY_FORMAT.get(expr, nfe.getMessage());
            throw new AciException(message);
        }
        if (valueAsInt < 0 || valueAsInt > 2359)
        {
            LocalizableMessage message = WARN_ACI_SYNTAX_INVALID_TIMEOFDAY_RANGE.get(expr);
            throw new AciException(message);
        }

        return new TimeOfDay(valueAsInt, type);
    }

    /**
     * Evaluates the timeofday bind rule using the evaluation context
     * passed into the method.
     * @param evalCtx  The evaluation context to use for the evaluation.
     * @return  An enumeration result representing the result of the
     * evaluation.
     */
    public EnumEvalResult evaluate(AciEvalContext evalCtx) {
        EnumEvalResult matched=EnumEvalResult.FALSE;

        int currentTime=TimeThread.getHourAndMinute();
        //check the type
        switch (type) {
        case EQUAL_BINDRULE_TYPE:
        case NOT_EQUAL_BINDRULE_TYPE:
            if (currentTime != timeRef)
            {
                matched=EnumEvalResult.TRUE;
            }
            break;

        case LESS_OR_EQUAL_BINDRULE_TYPE:
            if (currentTime <= timeRef)
            {
                matched=EnumEvalResult.TRUE;
            }
            break;

        case LESS_BINDRULE_TYPE:
            if (currentTime < timeRef)
            {
                matched=EnumEvalResult.TRUE;
            }
            break;

        case GREATER_OR_EQUAL_BINDRULE_TYPE:
            if (currentTime >= timeRef)
            {
                matched=EnumEvalResult.TRUE;
            }
            break;

        case GREATER_BINDRULE_TYPE:
            if (currentTime > timeRef)
            {
                matched=EnumEvalResult.TRUE;
            }
        }
        return matched.getRet(type, false);
    }

    /** {@inheritDoc} */
    @Override
    public String toString()
    {
        final StringBuilder sb = new StringBuilder();
        toString(sb);
        return sb.toString();
    }

    /** {@inheritDoc} */
    @Override
    public final void toString(StringBuilder buffer)
    {
        buffer.append(super.toString());
    }

}
