/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at legal-notices/CDDLv1_0.txt
 * or http://forgerock.org/license/CDDLv1.0.html.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at legal-notices/CDDLv1_0.txt.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2006-2008 Sun Microsystems, Inc.
 *      Portions Copyright 2013-2015 ForgeRock AS
 */
package org.opends.server.loggers;


import org.forgerock.i18n.LocalizableMessage;
import org.opends.messages.Severity;
import org.opends.server.admin.std.server.ErrorLogPublisherCfg;
import org.forgerock.opendj.config.server.ConfigException;
import org.opends.server.core.ServerContext;
import org.opends.server.types.DN;
import org.opends.server.types.InitializationException;
import org.opends.server.util.StaticUtils;
import org.opends.server.util.TimeThread;

/**
 * This class provides an implementation of an error logger where only messages
 * generated by a specified thread is actually logged.
 */
public class ThreadFilterTextErrorLogPublisher
    extends ErrorLogPublisher<ErrorLogPublisherCfg>
{
  private Thread thread;

  private TextWriter writer;

  /**
   * Construct a new instance with the provided settings.
   *
   * @param thread The thread to log from.
   * @param writer The writer used to write the messages.
   */
  public ThreadFilterTextErrorLogPublisher(Thread thread,
                                           TextWriter writer)
  {
    this.thread = thread;
    this.writer = writer;
  }

  /** {@inheritDoc} */
  @Override
  public void initializeLogPublisher(ErrorLogPublisherCfg config, ServerContext serverContext)
      throws ConfigException, InitializationException
  {
    // This class should only be used internally in the server and not be
    // configurable via the admin framework.
  }

  /** {@inheritDoc} */
  @Override
  public void close()
  {
    writer.shutdown();
  }

  /** {@inheritDoc} */
  @Override
  public void log(String category, Severity severity,
      LocalizableMessage message, Throwable exception)
  {
    if (message != null) {
      Thread currentThread = Thread.currentThread();
      if(this.thread.equals(currentThread) ||
          this.thread.getThreadGroup().equals(currentThread.getThreadGroup()))
      {
        StringBuilder sb = new StringBuilder();
        sb.append("[");
        sb.append(TimeThread.getLocalTime());
        sb.append("] category=").append(category).
        append(" severity=").append(severity).
        append(" msgID=").append(message.resourceName()).
        append("-").append(message.ordinal()).
        append(" msg=").append(message);
        if (exception != null)
        {
          sb.append(" exception=").append(
              StaticUtils.stackTraceToSingleLineString(exception));
        }

        this.writer.writeRecord(sb.toString());
      }
    }
  }

  /** {@inheritDoc} */
  @Override
  public boolean isEnabledFor(String category, Severity severity)
  {
    Thread currentThread = Thread.currentThread();
    return this.thread.equals(currentThread)
        || this.thread.getThreadGroup().equals(currentThread.getThreadGroup());
  }

  /** {@inheritDoc} */
  @Override
  public DN getDN()
  {
    // This class should only be used internally in the server and not be
    // configurable via the admin framework.
    return null;
  }
}
