/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at legal-notices/CDDLv1_0.txt
 * or http://forgerock.org/license/CDDLv1.0.html.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at legal-notices/CDDLv1_0.txt.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2008-2010 Sun Microsystems, Inc.
 *      Portions Copyright 2014-2015 ForgeRock AS
 */
package org.opends.server.monitors;

import java.util.Collections;
import java.util.List;

import org.forgerock.i18n.LocalizableMessage;
import org.forgerock.i18n.slf4j.LocalizedLogger;
import org.opends.server.admin.std.server.EntryCacheCfg;
import org.opends.server.admin.std.server.EntryCacheMonitorProviderCfg;
import org.opends.server.api.EntryCache;
import org.opends.server.api.MonitorProvider;
import org.opends.server.config.ConfigConstants;
import org.forgerock.opendj.config.server.ConfigException;
import org.opends.server.core.DirectoryServer;
import org.opends.server.types.Attribute;

import static org.opends.messages.ConfigMessages.*;

/**
 * This class defines a Directory Server monitor provider that can be used to
 * obtain information about the entry cache state. Note that the information
 * reported is obtained with no locking, so it may not be entirely consistent.
 */
public class EntryCacheMonitorProvider
       extends MonitorProvider<EntryCacheMonitorProviderCfg>
{

  private static final LocalizedLogger logger = LocalizedLogger.getLoggerForThisClass();

  /** The name for this monitor. */
  private String monitorName;

  /** The entry cache common name. */
  private String entryCacheName;

  /** The entry cache with which this monitor is associated. */
  private EntryCache<? extends EntryCacheCfg> entryCache;

  /** Entry cache monitor configuration. */
  private EntryCacheMonitorProviderCfg monitorConfiguration;

  /**
   * Creates default instance of this monitor provider.
   */
  public EntryCacheMonitorProvider()
  {
    this.entryCacheName = "Entry Caches";
    this.entryCache = DirectoryServer.getEntryCache();
  }

  /**
   * Creates implementation specific instance of this monitor provider.
   *
   * @param  entryCacheName  The name to use for this monitor provider.
   * @param  entryCache      The entry cache to associate this monitor
   *                         provider with.
   */
  public EntryCacheMonitorProvider(
    String entryCacheName,
    EntryCache<? extends EntryCacheCfg> entryCache)
  {
    this.entryCacheName = entryCacheName + " Entry Cache";
    this.entryCache = entryCache;
  }

  /** {@inheritDoc} */
  @Override
  public void initializeMonitorProvider(
    EntryCacheMonitorProviderCfg configuration)
    throws ConfigException
  {
    monitorName = entryCacheName;

    if (configuration != null) {
      monitorConfiguration = configuration;
    }
    if (monitorConfiguration == null) {
      LocalizableMessage message =
          INFO_WARN_CONFIG_ENTRYCACHE_NO_MONITOR_CONFIG_ENTRY.get(
              ConfigConstants.DN_ENTRY_CACHE_MONITOR_CONFIG, monitorName);
      logger.debug(message);
      throw new ConfigException(message);
    }
    if (!monitorConfiguration.isEnabled()) {
      LocalizableMessage message =
          INFO_WARN_CONFIG_ENTRYCACHE_MONITOR_CONFIG_DISABLED.get(
              ConfigConstants.DN_ENTRY_CACHE_MONITOR_CONFIG, monitorName);
      logger.debug(message);
      throw new ConfigException(message);
    }
  }

  /** {@inheritDoc} */
  @Override
  public String getMonitorInstanceName()
  {
    return monitorName;
  }

  /** {@inheritDoc} */
  @Override
  public List<Attribute> getMonitorData()
  {
    if (entryCache != null &&
        monitorConfiguration != null &&
        monitorConfiguration.isEnabled()) {
      // Get monitor data from the cache.
      return entryCache.getMonitorData();
    }
    return Collections.emptyList();
  }
}
