/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at legal-notices/CDDLv1_0.txt
 * or http://forgerock.org/license/CDDLv1.0.html.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at legal-notices/CDDLv1_0.txt.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2006-2009 Sun Microsystems, Inc.
 *      Portions Copyright 2013-2015 ForgeRock AS.
 */
package org.opends.server.types;

import java.util.List;
import java.util.Map;
import java.util.Set;

import org.forgerock.i18n.LocalizableMessage;
import org.forgerock.opendj.config.server.ConfigException;
import org.forgerock.opendj.ldap.ResultCode;
import org.forgerock.opendj.ldap.schema.MatchingRule;
import org.forgerock.opendj.ldap.schema.Syntax;
import org.opends.server.api.AlertGenerator;
import org.opends.server.api.ConfigHandler;
import org.opends.server.api.ExtendedOperationHandler;
import org.opends.server.api.InvokableComponent;
import org.opends.server.api.SASLMechanismHandler;
import org.opends.server.api.ServerShutdownListener;
import org.opends.server.config.ConfigEntry;
import org.opends.server.core.DirectoryServer;

import com.forgerock.opendj.util.OperatingSystem;

/**
 * This interface defines a set of methods that may be used by
 * third-party code to obtain information about the core Directory
 * Server configuration and the instances of various kinds of
 * components that have registered themselves with the server.
 * <BR><BR>
 * Note that this interface is not intended to be implemented by any
 * third-party code.  It is merely used to control which elements are
 * intended for use by external classes.
 */
@org.opends.server.types.PublicAPI(
     stability=org.opends.server.types.StabilityLevel.VOLATILE,
     mayInstantiate=false,
     mayExtend=false,
     mayInvoke=true)
public final class DirectoryConfig
{
  /**
   * Retrieves a reference to the Directory Server crypto manager.
   *
   * @return  A reference to the Directory Server crypto manager.
   */
  public static CryptoManager getCryptoManager()
  {
    return DirectoryServer.getCryptoManager();
  }



  /**
   * Retrieves the operating system on which the Directory Server is
   * running.
   *
   * @return  The operating system on which the Directory Server is
   *          running.
   */
  public static OperatingSystem getOperatingSystem()
  {
    return DirectoryServer.getOperatingSystem();
  }



  /**
   * Retrieves a reference to the Directory Server configuration
   * handler.
   *
   * @return  A reference to the Directory Server configuration
   *          handler.
   */
  public static ConfigHandler getConfigHandler()
  {
    return DirectoryServer.getConfigHandler();
  }



  /**
   * Retrieves the requested entry from the Directory Server
   * configuration.
   *
   * @param  entryDN  The DN of the configuration entry to retrieve.
   *
   * @return  The requested entry from the Directory Server
   *          configuration.
   *
   * @throws  ConfigException  If a problem occurs while trying to
   *                           retrieve the requested entry.
   */
  public static ConfigEntry getConfigEntry(DN entryDN)
         throws ConfigException
  {
    return DirectoryServer.getConfigEntry(entryDN);
  }



  /**
   * Retrieves the path to the root directory for this instance of the
   * Directory Server.
   *
   * @return  The path to the root directory for this instance of the
   *          Directory Server.
  */
  public static String getServerRoot()
  {
    return DirectoryServer.getServerRoot();
  }



  /**
   * Retrieves the time that the Directory Server was started, in
   * milliseconds since the epoch.
   *
   * @return  The time that the Directory Server was started, in
   *          milliseconds since the epoch.
   */
  public static long getStartTime()
  {
    return DirectoryServer.getStartTime();
  }



  /**
   * Retrieves the time that the Directory Server was started,
   * formatted in UTC.
   *
   * @return  The time that the Directory Server was started,
   *          formatted in UTC.
   */
  public static String getStartTimeUTC()
  {
    return DirectoryServer.getStartTimeUTC();
  }



  /**
   * Retrieves a reference to the Directory Server schema.
   *
   * @return  A reference to the Directory Server schema.
   */
  public static Schema getSchema()
  {
    return DirectoryServer.getSchema();
  }



  /**
   * Retrieves the set of matching rules registered with the Directory
   * Server.  The mapping will be between the lowercase name or OID
   * for each matching rule and the matching rule implementation.  The
   * same matching rule instance may be included multiple times with
   * different keys.  The returned map must not be altered by the
   * caller.
   *
   * @return  The set of matching rules registered with the Directory
   *          Server.
   */
  public static Map<String,MatchingRule> getMatchingRules()
  {
    return DirectoryServer.getMatchingRules();
  }



  /**
   * Retrieves the matching rule with the specified name or OID.
   *
   * @param  lowerName  The lowercase name or OID for the matching
   *                    rule to retrieve.
   *
   * @return  The requested matching rule, or <CODE>null</CODE> if no
   *          such matching rule has been defined in the server.
   */
  public static MatchingRule getMatchingRule(String lowerName)
  {
    return DirectoryServer.getMatchingRule(lowerName);
  }



  /**
   * Retrieves the approximate matching rule with the specified name
   * or OID.
   *
   * @param  lowerName  The lowercase name or OID for the approximate
   *                    matching rule to retrieve.
   *
   * @return  The requested approximate matching rule, or
   *          <CODE>null</CODE> if no such matching rule has been
   *          defined in the server.
   */
  public static MatchingRule
       getApproximateMatchingRule(String lowerName)
  {
    return DirectoryServer.getMatchingRule(lowerName);
  }



  /**
   * Retrieves the equality matching rule with the specified name or
   * OID.
   *
   * @param  lowerName  The lowercase name or OID for the equality
   *                    matching rule to retrieve.
   *
   * @return  The requested equality matching rule, or
   *          <CODE>null</CODE> if no such matching rule has been
   *          defined in the server.
   */
  public static MatchingRule
       getEqualityMatchingRule(String lowerName)
  {
    return DirectoryServer.getMatchingRule(lowerName);
  }



  /**
   * Retrieves the ordering matching rule with the specified name or
   * OID.
   *
   * @param  lowerName  The lowercase name or OID for the ordering
   *                    matching rule to retrieve.
   *
   * @return  The requested ordering matching rule, or
   *          <CODE>null</CODE> if no such matching rule has been
   *          defined in the server.
   */
  public static MatchingRule
       getOrderingMatchingRule(String lowerName)
  {
    return DirectoryServer.getMatchingRule(lowerName);
  }



  /**
   * Retrieves the substring matching rule with the specified name or
   * OID.
   *
   * @param  lowerName  The lowercase name or OID for the substring
   *                    matching rule to retrieve.
   *
   * @return  The requested substring matching rule, or
   *          <CODE>null</CODE> if no such matching rule has been
   *          defined in the server.
   */
  public static MatchingRule
       getSubstringMatchingRule(String lowerName)
  {
    return DirectoryServer.getMatchingRule(lowerName);
  }



  /**
   * Retrieves the set of objectclasses registered with the Directory
   * Server.  The mapping will be between the lowercase name or OID
   * for each objectclass and the objectclass implementation.  The
   * same objectclass instance may be included multiple times with
   * different keys.  The returned map must not be altered by the
   * caller.
   *
   * @return  The set of objectclasses defined in the Directory
   *          Server.
   */
  public static Map<String,ObjectClass> getObjectClasses()
  {
    return DirectoryServer.getObjectClasses();
  }



  /**
   * Retrieves the objectclass for the provided lowercase name or OID.
   * It can optionally return a generated "default" version if the
   * requested objectclass is not defined in the schema.
   *
   * @param  lowerName      The lowercase name or OID for the
   *                        objectclass to retrieve.
   * @param  returnDefault  Indicates whether to generate a default
   *                        version if the requested objectclass is
   *                        not defined in the server schema.
   *
   * @return  The objectclass type, or <CODE>null</CODE> if there is
   *          no objectclass with the specified name or OID defined in
   *          the server schema and a default class should not be
   *          returned.
   */
  public static ObjectClass
       getObjectClass(String lowerName, boolean returnDefault)
  {
    return DirectoryServer.getObjectClass(lowerName, returnDefault);
  }



  /**
   * Retrieves the "top" objectClass, which should be the topmost
   * objectclass in the inheritance chain for most other
   * objectclasses.
   *
   * @return  The "top" objectClass.
   */
  public static ObjectClass getTopObjectClass()
  {
    return DirectoryServer.getTopObjectClass();
  }



  /**
   * Retrieves the set of attribute type definitions that have been
   * defined in the Directory Server.  The mapping will be between the
   * lowercase name or OID for each attribute type and the attribute
   * type implementation.  The same attribute type may be included
   * multiple times with different keys.  The returned map must not be
   * altered by the caller.
   *
   * @return The set of attribute type definitions that have been
   *         defined in the Directory Server.
   */
  public static Map<String,AttributeType> getAttributeTypes()
  {
    return DirectoryServer.getAttributeTypes();
  }

  /**
   * Retrieves the attribute type for the "objectClass" attribute.
   *
   * @return  The attribute type for the "objectClass" attribute.
   */
  public static AttributeType getObjectClassAttributeType()
  {
    return DirectoryServer.getObjectClassAttributeType();
  }



  /**
   * Retrieves the set of attribute syntaxes defined in the Directory
   * Server.  The mapping will be between the OID and the
   * corresponding syntax implementation.  The returned map must not
   * be altered by the caller.
   *
   * @return  The set of attribute syntaxes defined in the Directory
   *          Server.
   */
  public static Map<String,Syntax>
       getAttributeSyntaxes()
  {
    return DirectoryServer.getAttributeSyntaxes();
  }

  /**
   * Retrieves the default attribute syntax that should be used for
   * attributes that are not defined in the server schema.
   *
   * @return  The default attribute syntax that should be used for
   *          attributes that are not defined in the server schema.
   */
  public static Syntax getDefaultAttributeSyntax()
  {
    return DirectoryServer.getDefaultAttributeSyntax();
  }



  /**
   * Retrieves the default attribute syntax that should be used for
   * attributes that are not defined in the server schema and are
   * meant to store binary values.
   *
   * @return  The default attribute syntax that should be used for
   *          attributes that are not defined in the server schema and
   *          are meant to store binary values.
   */
  public static Syntax getDefaultBinarySyntax()
  {
    return DirectoryServer.getDefaultBinarySyntax();
  }



  /**
   * Retrieves the default attribute syntax that should be used for
   * attributes that are not defined in the server schema and are
   * meant to store Boolean values.
   *
   * @return  The default attribute syntax that should be used for
   *          attributes that are not defined in the server schema and
   *          are meant to store Boolean values.
   */
  public static Syntax getDefaultBooleanSyntax()
  {
    return DirectoryServer.getDefaultBooleanSyntax();
  }



  /**
   * Retrieves the default attribute syntax that should be used for
   * attributes that are not defined in the server schema and are
   * meant to store DN values.
   *
   * @return  The default attribute syntax that should be used for
   *          attributes that are not defined in the server schema and
   *          are meant to store DN values.
   */
  public static Syntax getDefaultDNSyntax()
  {
    return DirectoryServer.getDefaultDNSyntax();
  }



  /**
   * Retrieves the default attribute syntax that should be used for
   * attributes that are not defined in the server schema and are
   * meant to store integer values.
   *
   * @return  The default attribute syntax that should be used for
   *          attributes that are not defined in the server schema and
   *          are meant to store integer values.
   */
  public static Syntax getDefaultIntegerSyntax()
  {
    return DirectoryServer.getDefaultIntegerSyntax();
  }



  /**
   * Retrieves the default attribute syntax that should be used for
   * attributes that are not defined in the server schema and are
   * meant to store string values.
   *
   * @return  The default attribute syntax that should be used for
   *          attributes that are not defined in the server schema and
   *          are meant to store string values.
   */
  public static Syntax getDefaultStringSyntax()
  {
    return DirectoryServer.getDefaultStringSyntax();
  }



  /**
   * Retrieves the set of matching rule uses defined in the Directory
   * Server.  The mapping will be between the matching rule and its
   * corresponding matching rule use.  The returned map must not be
   * altered by the caller.
   *
   * @return  The set of matching rule uses defined in the Directory
   *          Server.
   */
  public static Map<MatchingRule,MatchingRuleUse>
       getMatchingRuleUses()
  {
    return DirectoryServer.getMatchingRuleUses();
  }



  /**
   * Retrieves the matching rule use associated with the provided
   * matching rule.
   *
   * @param  matchingRule  The matching rule for which to retrieve the
   *                       matching rule use.
   *
   * @return  The matching rule use for the provided matching rule, or
   *          <CODE>null</CODE> if none is defined.
   */
  public static MatchingRuleUse
       getMatchingRuleUse(MatchingRule matchingRule)
  {
    return DirectoryServer.getMatchingRuleUse(matchingRule);
  }



  /**
   * Retrieves the set of DIT content rules defined in the Directory
   * Server.  The mapping will be between the structural objectclass
   * and its corresponding DIT content rule.  The returned map must
   * not be altered by the caller.
   *
   * @return  The set of DIT content rules defined in the Directory
   *          Server.
   */
  public static Map<ObjectClass,DITContentRule>
       getDITContentRules()
  {
    return DirectoryServer.getDITContentRules();
  }



  /**
   * Retrieves the DIT content rule associated with the specified
   * objectclass.
   *
   * @param  objectClass  The objectclass for which to retrieve the
   *                      associated DIT content rule.
   *
   * @return  The requested DIT content rule, or <CODE>null</CODE> if
   *          no such rule is defined in the schema.
   */
  public static DITContentRule
       getDITContentRule(ObjectClass objectClass)
  {
    return DirectoryServer.getDITContentRule(objectClass);
  }



  /**
   * Retrieves the set of DIT structure rules defined in the Directory
   * Server.  The mapping will be between the name form and its
   * corresponding DIT structure rule.  The returned map must not be
   * altered by the caller.
   *
   * @return  The set of DIT structure rules defined in the Directory
   *          Server.
   */
  public static Map<NameForm,DITStructureRule>
       getDITStructureRules()
  {
    return DirectoryServer.getDITStructureRules();
  }



  /**
   * Retrieves the DIT structure rule associated with the provided
   * rule ID.
   *
   * @param  ruleID  The rule ID for which to retrieve the associated
   *                 DIT structure rule.
   *
   * @return  The requested DIT structure rule, or <CODE>null</CODE>
   *          if no such rule is defined.
   */
  public static DITStructureRule getDITStructureRule(int ruleID)
  {
    return DirectoryServer.getDITStructureRule(ruleID);
  }



  /**
   * Retrieves the DIT structure rule associated with the provided
   * name form.
   *
   * @param  nameForm  The name form for which to retrieve the
   *                   associated DIT structure rule.
   *
   * @return  The requested DIT structure rule, or <CODE>null</CODE>
   *          if no such rule is defined.
   */
  public static DITStructureRule
       getDITStructureRule(NameForm nameForm)
  {
    return DirectoryServer.getDITStructureRule(nameForm);
  }



  /**
   * Retrieves the set of name forms defined in the Directory Server.
   * The mapping will be between the structural objectclass and its
   * corresponding name forms.  The returned map must not be altered
   * by the caller.
   *
   * @return  The set of name forms defined in the Directory Server.
   */
  public static Map<ObjectClass,List<NameForm>> getNameForms()
  {
    return DirectoryServer.getNameForms();
  }



  /**
   * Retrieves the list of name forms associated with the specified
   * structural objectclass.
   *
   * @param  objectClass  The structural objectclass for which to
   *                      retrieve the  associated name form.
   *
   * @return  The list of requested name forms, or <CODE>null</CODE>
   *           if no such name form is defined in the schema.
   */
  public static List<NameForm> getNameForm(ObjectClass objectClass)
  {
    return DirectoryServer.getNameForm(objectClass);
  }



  /**
   * Retrieves the name form associated with the specified name or
   * OID.
   *
   * @param  lowerName  The name or OID of the name form to retrieve,
   *                    formatted in all lowercase characters.
   *
   * @return  The requested name form, or <CODE>null</CODE> if no such
   *          name form is defined in the schema.
   */
  public static NameForm getNameForm(String lowerName)
  {
    return DirectoryServer.getNameForm(lowerName);
  }



  /**
   * Registers the provided invokable component with the Directory
   * Server.
   *
   * @param  component  The invokable component to register.
   */
  public static void registerInvokableComponent(
                                InvokableComponent component)
  {
    DirectoryServer.registerInvokableComponent(component);
  }



  /**
   * Deregisters the provided invokable component with the Directory
   * Server.
   *
   * @param  component  The invokable component to deregister.
   */
  public static void deregisterInvokableComponent(
                                InvokableComponent component)
  {
    DirectoryServer.deregisterInvokableComponent(component);
  }



  /**
   * Registers the provided alert generator with the Directory Server.
   *
   * @param  alertGenerator  The alert generator to register.
   */
  public static void registerAlertGenerator(
                                AlertGenerator alertGenerator)
  {
    DirectoryServer.registerAlertGenerator(alertGenerator);
  }



  /**
   * Deregisters the provided alert generator with the Directory
   * Server.
   *
   * @param  alertGenerator  The alert generator to deregister.
   */
  public static void deregisterAlertGenerator(
                                AlertGenerator alertGenerator)
  {
    DirectoryServer.deregisterAlertGenerator(alertGenerator);
  }



  /**
   * Sends an alert notification with the provided information.
   *
   * @param  generator     The alert generator that created the alert.
   * @param  alertType     The alert type name for this alert.
   * @param  alertMessage  A message (possibly <CODE>null</CODE>) that
   *                       can provide more information about this
   *                       alert.
   */
  public static void
       sendAlertNotification(AlertGenerator generator,
                             String alertType,
                             LocalizableMessage alertMessage)
  {
    DirectoryServer.sendAlertNotification(generator, alertType,
            alertMessage);
  }



  /**
   * Retrieves the result code that should be used when the Directory
   * Server encounters an internal server error.
   *
   * @return  The result code that should be used when the Directory
   *          Server encounters an internal server error.
   */
  public static ResultCode getServerErrorResultCode()
  {
    return DirectoryServer.getServerErrorResultCode();
  }



  /**
   * Retrieves the entry with the requested DN.  It will first
   * determine which backend should be used for this DN and will then
   * use that backend to retrieve the entry.  The caller must already
   * hold the appropriate lock on the specified entry.
   *
   * @param  entryDN  The DN of the entry to retrieve.
   *
   * @return  The requested entry, or <CODE>null</CODE> if it does not
   *          exist.
   *
   * @throws  DirectoryException  If a problem occurs while attempting
   *                              to retrieve the entry.
   */
  public static Entry getEntry(DN entryDN)
         throws DirectoryException
  {
    return DirectoryServer.getEntry(entryDN);
  }



  /**
   * Indicates whether the specified entry exists in the Directory
   * Server.  The caller is not required to hold any locks when
   * invoking this method.
   *
   * @param  entryDN  The DN of the entry for which to make the
   *                  determination.
   *
   * @return  <CODE>true</CODE> if the specified entry exists in one
   *          of the backends, or <CODE>false</CODE> if it does not.
   *
   * @throws  DirectoryException  If a problem occurs while attempting
   *                              to make the determination.
   */
  public static boolean entryExists(DN entryDN)
         throws DirectoryException
  {
    return DirectoryServer.entryExists(entryDN);
  }



  /**
   * Retrieves the set of OIDs for the supported controls registered
   * with the Directory Server.
   *
   * @return  The set of OIDS for the supported controls registered
   *          with the Directory Server.
   */
  public static Set<String> getSupportedControls()
  {
    return DirectoryServer.getSupportedControls();
  }



  /**
   * Indicates whether the specified OID is registered with the
   * Directory Server as a supported control.
   *
   * @param  controlOID  The OID of the control for which to make the
   *                     determination.
   *
   * @return  <CODE>true</CODE> if the specified OID is registered
   *          with the server as a supported control, or
   *          <CODE>false</CODE> if not.
   */
  public static boolean isSupportedControl(String controlOID)
  {
    return DirectoryServer.isSupportedControl(controlOID);
  }



  /**
   * Registers the provided OID as a supported control for the
   * Directory Server.  This will have no effect if the specified
   * control OID is already present in the list of supported controls.
   *
   * @param  controlOID  The OID of the control to register as a
   *                     supported control.
   */
  public static void registerSupportedControl(String controlOID)
  {
    DirectoryServer.registerSupportedControl(controlOID);
  }



  /**
   * Deregisters the provided OID as a supported control for the
   * Directory Server.  This will have no effect if the specified
   * control OID is not present in the list of supported controls.
   *
   * @param  controlOID  The OID of the control to deregister as a
   *                     supported control.
   */
  public static void
       deregisterSupportedControl(String controlOID)
  {
    DirectoryServer.deregisterSupportedControl(controlOID);
  }



  /**
   * Retrieves the set of OIDs for the supported features registered
   * with the Directory Server.
   *
   * @return  The set of OIDs for the supported features registered
   *          with the Directory Server.
   */
  public static Set<String> getSupportedFeatures()
  {
    return DirectoryServer.getSupportedFeatures();
  }



  /**
   * Indicates whether the specified OID is registered with the
   * Directory Server as a supported feature.
   *
   * @param  featureOID  The OID of the feature for which to make the
   *                     determination.
   *
   * @return  <CODE>true</CODE> if the specified OID is registered
   *          with the server as a supported feature, or
   *          <CODE>false</CODE> if not.
   */
  public static boolean isSupportedFeature(String featureOID)
  {
    return DirectoryServer.isSupportedFeature(featureOID);
  }



  /**
   * Registers the provided OID as a supported feature for the
   * Directory Server.  This will have no effect if the specified
   * feature OID is already present in the list of supported features.
   *
   * @param  featureOID  The OID of the feature to register as a
   *                     supported feature.
   */
  public static void registerSupportedFeature(String featureOID)
  {
    DirectoryServer.registerSupportedFeature(featureOID);
  }



  /**
   * Deregisters the provided OID as a supported feature for the
   * Directory Server.  This will have no effect if the specified
   * feature OID is not present in the list of supported features.
   *
   * @param  featureOID  The OID of the feature to deregister as a
   *                     supported feature.
   */
  public static void
       deregisterSupportedFeature(String featureOID)
  {
    DirectoryServer.deregisterSupportedFeature(featureOID);
  }



  /**
   * Retrieves the set of extended operations that may be processed by
   * the Directory Server.  The mapping will be between the OID and
   * the extended operation handler providing the logic for the
   * extended operation with that OID.  The returned map must not be
   * altered by the caller.
   *
   * @return  The set of extended operations that may be processed by
   *          the Directory Server.
   */
  public static Map<String,ExtendedOperationHandler>
                     getSupportedExtensions()
  {
    return DirectoryServer.getSupportedExtensions();
  }



  /**
   * Retrieves the handler for the extended operation for the provided
   * extended operation OID.
   *
   * @param  oid  The OID of the extended operation to retrieve.
   *
   * @return  The handler for the specified extended operation, or
   *          <CODE>null</CODE> if there is none.
   */
  public static ExtendedOperationHandler
       getExtendedOperationHandler(String oid)
  {
    return DirectoryServer.getExtendedOperationHandler(oid);
  }



  /**
   * Registers the provided extended operation handler with the
   * Directory Server.
   *
   * @param  oid      The OID for the extended operation to register.
   * @param  handler  The extended operation handler to register with
   *                  the Directory Server.
   */
  public static void registerSupportedExtension(String oid,
                          ExtendedOperationHandler handler)
  {
    DirectoryServer.registerSupportedExtension(oid, handler);
  }



  /**
   * Deregisters the provided extended operation handler with the
   * Directory Server.
   *
   * @param  oid  The OID for the extended operation to deregister.
   */
  public static void deregisterSupportedExtension(String oid)
  {
    DirectoryServer.deregisterSupportedExtension(oid);
  }



  /**
   * Retrieves the set of SASL mechanisms that are supported by the
   * Directory Server.  The mapping will be between the mechanism name
   * and the SASL mechanism handler that implements support for that
   * mechanism.  The returned map must not be altered by the caller.
   *
   * @return  The set of SASL mechanisms that are supported by the
   *          Directory Server.
   */
  public static Map<String,SASLMechanismHandler>
                     getSupportedSASLMechanisms()
  {
    return DirectoryServer.getSupportedSASLMechanisms();
  }



  /**
   * Retrieves the handler for the specified SASL mechanism.
   *
   * @param  name  The name of the SASL mechanism to retrieve.
   *
   * @return  The handler for the specified SASL mechanism, or
   *          <CODE>null</CODE> if there is none.
   */
  public static SASLMechanismHandler
       getSASLMechanismHandler(String name)
  {
    return DirectoryServer.getSASLMechanismHandler(name);
  }



  /**
   * Registers the provided SASL mechanism handler with the Directory
   * Server.
   *
   * @param  name     The name of the SASL mechanism to be registered.
   * @param  handler  The SASL mechanism handler to register with the
   *                  Directory Server.
   */
  public static void
       registerSASLMechanismHandler(String name,
                                    SASLMechanismHandler handler)
  {
    DirectoryServer.registerSASLMechanismHandler(name, handler);
  }



  /**
   * Deregisters the provided SASL mechanism handler with the
   * Directory Server.
   *
   * @param  name  The name of the SASL mechanism to be deregistered.
   */
  public static void deregisterSASLMechanismHandler(String name)
  {
    DirectoryServer.deregisterSASLMechanismHandler(name);
  }

  /**
   * Registers the provided shutdown listener with the Directory
   * Server so that it will be notified when the server shuts down.
   *
   * @param  listener  The shutdown listener to register with the
   *                   Directory Server.
   */
  public static void
       registerShutdownListener(ServerShutdownListener listener)
  {
    DirectoryServer.registerShutdownListener(listener);
  }



  /**
   * Deregisters the provided shutdown listener with the Directory
   * Server.
   *
   * @param  listener  The shutdown listener to deregister with the
   *                   Directory Server.
   */
  public static void
       deregisterShutdownListener(ServerShutdownListener listener)
  {
    DirectoryServer.deregisterShutdownListener(listener);
  }



  /**
   * Retrieves the full version string for the Directory Server.
   *
   * @return  The full version string for the Directory Server.
   */
  public static String getVersionString()
  {
    return DirectoryServer.getVersionString();
  }
}

