/*
 * CDDL HEADER START
 *
 * The contents of this file are subject to the terms of the
 * Common Development and Distribution License, Version 1.0 only
 * (the "License").  You may not use this file except in compliance
 * with the License.
 *
 * You can obtain a copy of the license at legal-notices/CDDLv1_0.txt
 * or http://forgerock.org/license/CDDLv1.0.html.
 * See the License for the specific language governing permissions
 * and limitations under the License.
 *
 * When distributing Covered Code, include this CDDL HEADER in each
 * file and include the License file at legal-notices/CDDLv1_0.txt.
 * If applicable, add the following below this CDDL HEADER, with the
 * fields enclosed by brackets "[]" replaced with your own identifying
 * information:
 *      Portions Copyright [yyyy] [name of copyright owner]
 *
 * CDDL HEADER END
 *
 *
 *      Copyright 2008 Sun Microsystems, Inc.
 *      Portions Copyright 2012-2015 ForgeRock AS.
 */

package org.opends.server.admin;

import static org.testng.Assert.*;

import java.util.Comparator;
import java.util.List;

import org.opends.server.admin.std.meta.RootCfgDefn;
import org.opends.server.DirectoryServerTestCase;
import org.testng.annotations.BeforeClass;
import org.testng.annotations.DataProvider;
import org.testng.annotations.Test;

/**
 * ClassPropertyDefinition Tester.
 */
public class ClassPropertyDefinitionTest extends DirectoryServerTestCase {

  /** Dummy class for testing. */
  public static final class Dummy
  {
    public class X {}
    public Comparator<Dummy> comparator()
    {
      return new Comparator<ClassPropertyDefinitionTest.Dummy>()
      {
        public int compare(Dummy o1, Dummy o2)
        {
          // No implementation required.
          return 0;
        }
      };
    }
  }

  ClassPropertyDefinition.Builder builder;

  /**
   * Sets up tests.
   */
  @BeforeClass
  public void setUp() {
    ClassPropertyDefinition.setAllowClassValidation(true);
  }

  /**
   * @return data for testing
   */
  @DataProvider(name = "testBuilderAddInstanceOf")
  public Object[][] createBuilderAddInstanceOfData() {
    return new Object[][]{
            { "org.opends.server.admin.ClassPropertyDefinitionTest" },
            { "org.opends.server.admin.ClassPropertyDefinitionTest$Dummy" },
            { "org.opends.server.admin.ClassPropertyDefinitionTest$Dummy$X" },
            { "org.opends.server.admin.ClassPropertyDefinitionTest$Dummy$1" },
    };
  }

  /**
   * Tests builder.addInstanceOf with valid data
   * @param className name of class name to add
   */
  @Test(dataProvider = "testBuilderAddInstanceOf")
  public void testBuilderAddInstanceOf(String className) {
    ClassPropertyDefinition.Builder localBuilder =
            ClassPropertyDefinition.createBuilder(RootCfgDefn.getInstance(),
                "test-property");
    localBuilder.addInstanceOf(className);
    ClassPropertyDefinition cpd = localBuilder.getInstance();
    List<String> instances = cpd.getInstanceOfInterface();
    assertTrue(instances.contains(className));
  }

  /**
   * @return data for testing with illegal values
   */
  @DataProvider(name = "testBuilderAddInstanceOf2")
  public Object[][] createBuilderAddInstanceOfData2() {
    return new Object[][]{
            { "1" },
            { "" },
            { " " },
            { "  " },
            { "abc." },
            { "abc.123" },
            { "abc.123$" },
    };
  }

  /**
   * Tests builder.addInstanceOf with valid data
   * @param className name of class to add
   */
  @Test(dataProvider = "testBuilderAddInstanceOf2",
          expectedExceptions = {IllegalArgumentException.class})
  public void testBuilderAddInstanceOf2(String className) {
    ClassPropertyDefinition.Builder localBuilder =
            ClassPropertyDefinition.createBuilder(
                RootCfgDefn.getInstance(), "test-property");
    localBuilder.addInstanceOf(className);
    ClassPropertyDefinition cpd = localBuilder.getInstance();
    List<String> instances = cpd.getInstanceOfInterface();
    assertTrue(instances.contains(className));
  }

  /**
   * @return data for testing with illegal values
   */
  @DataProvider(name = "testLoadClassData")
  public Object[][] createTestLoadData() {
    return new Object[][]{
            { "java.io.Serializable",
                    "java.lang.String",
                    Object.class,
                    String.class },
            { "java.io.Serializable",
                    "java.lang.String",
                    String.class,
                    String.class },
            { "java.lang.Number", // abstract class
                    "java.lang.Long",
                    Number.class,
                    Long.class },
    };
  }

  @Test(dataProvider = "testLoadClassData")
  public <T> void testLoadClass(String interfaceName, String loadClassName,
                            Class<T> instanceOfClass, Class expectedClass) {
    ClassPropertyDefinition.Builder localBuilder =
            ClassPropertyDefinition.createBuilder(
                RootCfgDefn.getInstance(), "test-property");
    localBuilder.addInstanceOf(interfaceName);
    ClassPropertyDefinition cpd = localBuilder.getInstance();
    Class clazz = cpd.loadClass(loadClassName, instanceOfClass);
    assertEquals(clazz, expectedClass);
  }

  /**
   * @return data for testing with illegal values
   */
  @DataProvider(name = "testLoadClassData2")
  public Object[][] createTestLoadData2() {
    return new Object[][]{
            { "java.lang.Runnable",
                    "java.lang.String",
                    Object.class,
                    String.class },
            { "java.lang.Runnable",
                    "some.bogus.ClassName",
                    Object.class,
                    String.class },
            { "java.lang.Runnable",
                    "java.lang.String",
                    Number.class,
                    Number.class },
    };
  }

  @Test(dataProvider = "testLoadClassData2",
          expectedExceptions = {PropertyException.class})
  public <T> void testLoadClass2(String interfaceName, String loadClassName,
                            Class<T> instanceOfClass, Class expectedClass) {
    ClassPropertyDefinition.Builder localBuilder =
            ClassPropertyDefinition.createBuilder(
                RootCfgDefn.getInstance(), "test-property");
    localBuilder.addInstanceOf(interfaceName);
    ClassPropertyDefinition cpd = localBuilder.getInstance();
    Class clazz = cpd.loadClass(loadClassName, instanceOfClass);
    assertEquals(clazz, String.class);
  }

}
