/**
 * @file Copyright (c) 2011 jdknight
 *
 * Permission is hereby granted, free of charge, to any person obtaining a copy
 * of this software and associated documentation files (the "Software"), to deal
 * in the Software without restriction, including without limitation the rights
 * to use, copy, modify, merge, publish, distribute, sublicense, and/or sell
 * copies of the Software, and to permit persons to whom the Software is
 * furnished to do so, subject to the following conditions:
 *
 * The above copyright notice and this permission notice shall be included in
 * all copies or substantial portions of the Software.
 *
 * THE SOFTWARE IS PROVIDED "AS IS", WITHOUT WARRANTY OF ANY KIND, EXPRESS OR
 * IMPLIED, INCLUDING BUT NOT LIMITED TO THE WARRANTIES OF MERCHANTABILITY,
 * FITNESS FOR A PARTICULAR PURPOSE AND NONINFRINGEMENT. IN NO EVENT SHALL THE
 * AUTHORS OR COPYRIGHT HOLDERS BE LIABLE FOR ANY CLAIM, DAMAGES OR OTHER
 * LIABILITY, WHETHER IN AN ACTION OF CONTRACT, TORT OR OTHERWISE, ARISING FROM,
 * OUT OF OR IN CONNECTION WITH THE SOFTWARE OR THE USE OR OTHER DEALINGS IN THE
 * SOFTWARE.
 */
package de.javastream.java.net.jmac;

import de.javastream.java.net.IpCache;
import java.io.BufferedReader;
import java.io.IOException;
import java.io.InputStreamReader;
import java.util.List;
import java.util.logging.Logger;
import java.util.regex.Matcher;
import java.util.regex.Pattern;

/**
 * A MAC cache request.
 *
 * <p></p>
 * An example of retrieving a remote addresses MAC address:
 * <blockquote><pre>
 * String ipAddress = ?;
 *
 * String macAddress = MacCache.get(ipAddress);
 * if(macAddress != null)
 * {
 *    System.out.println("Found MAC: " + macAddress);
 * }
 * else
 * {
 *    System.out.println("Cannot find MAC address for the provided address.");
 * }
 * </pre></blockquote>
 */
public final class MacCache {

    private final static Logger LOG = Logger.getLogger(MacCache.class.getSimpleName());
    private static MacCache instance = null;
    private static final String ETHER = "ether";
    private static final String INET = "inet";
    private List<String> localIps;

    /**
     * Private constructor.
     */
    private MacCache() {
        localIps = IpCache.getInstance().getLocalIpv4s();
    }

    public static MacCache getInstance() {
        if (instance == null) {
            instance = new MacCache();
        }
        return instance;
    }

    /**
     * Return the MAC address of a currently native-cached local address.
     *
     * <p>
     * The returned MAC address, if not null, will always return leading-zeroed
     * for each group of two (2) hexadecimal digits with the separator always
     * being a dash (-).
     *
     * For example: ff-00-ff-00-ff-00
     * </p>
     *
     * <p>
     * This method is non-blocking.
     * </p>
     *
     * @param localAddress The address.
     * @return The MAC address of the provided address; null, if not cached.
     *
     * @throws UnsupportedOperatingSystemException Thrown if the current
     * operating system is not supported.
     */
    public String getMacFromAddress(String localAddress) throws UnsupportedOperatingSystemException {
        String foundMacAddress = null;
        String operatingSystemName = System.getProperty("os.name");
        String[] arpCommand;
        if (operatingSystemName.startsWith("Windows") == true) {
            arpCommand = new String[]{"arp", "-a", localAddress};
        } else if (operatingSystemName.startsWith("Linux") == true
                || operatingSystemName.startsWith("HP-UX") == true) {
            arpCommand = new String[]{"arp", "-n", "-a", localAddress};
        } else if (operatingSystemName.startsWith("Mac") == true) {
            if (localIps.contains(localAddress)) {
                return getMacAddressForMac(localAddress);
            }
            arpCommand = new String[]{"arp", "-n", localAddress};
        } else {
            // Note: Unsupported system.
            throw new UnsupportedOperatingSystemException("The current operating system '" + operatingSystemName + "' is not supported.");
        }

        // Example patterns:
        //  Windows/*nix | 00-00-00-00-00-00 to ff-ff-ff-ff-ff-ff
        //  OSX          | 0:0:0:0:0:0 to ff:ff:ff:ff:ff:ff
        Pattern macAddressPattern = Pattern.compile("([a-fA-F0-9]{1,2}(-|:)){5}[a-fA-F0-9]{1,2}");
        try {
            Process process = Runtime.getRuntime().exec(arpCommand);
            process.getOutputStream().close();

            BufferedReader inputBuffer = new BufferedReader(new InputStreamReader(process.getInputStream()));

            // Scan output for a matching MAC address.
            String rawOutput;
            while ((rawOutput = inputBuffer.readLine()) != null) {
                Matcher matcher = macAddressPattern.matcher(rawOutput);
                if (matcher.find() == true) {
                    foundMacAddress = matcher.group();
                    break;
                }
            }
        } catch (IOException e) {
            LOG.warning(e.getLocalizedMessage());
        }



        return normalize(foundMacAddress);
    }

    private String normalize(String foundMacAddress) {
        if (foundMacAddress != null) {
            // Always return '-' spacer.
            foundMacAddress = foundMacAddress.replace(':', '-');

            // Always return leading-zeroed portions.
            if (foundMacAddress.length() != 17) {
                String[] addressPortions = foundMacAddress.split("-");
                StringBuilder buffer = new StringBuilder();

                int i = 0;
                do {
                    String currentPortion = addressPortions[i];
                    if (currentPortion.length() == 1) {
                        buffer.append("0");
                    }
                    buffer.append(currentPortion);

                    i++;
                    if (i == addressPortions.length) {
                        break;
                    }

                    buffer.append("-");
                } while (true);

                foundMacAddress = buffer.toString();
            }
        }
        return foundMacAddress;
    }

    private String getMacAddressForMac(String localAddress) {
        String ether, rawOutput;
        ether = null;
        String[] arpCommand = new String[]{"ifconfig", "-a"};
        try {
            Process process = Runtime.getRuntime().exec(arpCommand);
            process.getOutputStream().close();
            BufferedReader inputBuffer = new BufferedReader(new InputStreamReader(process.getInputStream()));
            while ((rawOutput = inputBuffer.readLine()) != null) {
                if (rawOutput.trim().startsWith(ETHER)) {
                    ether = rawOutput.replace(ETHER, "").trim();
                }
                if (rawOutput.trim().startsWith(INET) && rawOutput.replace(INET, "").trim().startsWith(localAddress)) {
                    return normalize(ether);
                }
            }
        } catch (IOException e) {
            LOG.warning(e.getLocalizedMessage());
        }
        return null;
    }

    /**
     * Return the MAC address of a currently native-cached local address.
     *
     * <p>
     * The returned MAC address, if not null, will always return leading-zeroed
     * for each group of two (2) hexadecimal digits with the separator always
     * being a dash (-).
     *
     * For example: ff-00-ff-00-ff-00
     * </p>
     *
     * <p>
     * This method is non-blocking.
     * </p>
     *
     * @param localAddress The address.
     * @return The MAC address of the provided address; null, if not cached or
     * system is not supported.
     */
    public static String get(String localAddress) {
        try {
            return getInstance().getMacFromAddress(localAddress);
        } catch (UnsupportedOperatingSystemException e) {
            return null;
        }
    }
}
