/**
 * IpCache 14.04.2013
 *
 * @author Philipp Haussleiter
 *
 */
package de.javastream.java.net;

import java.io.IOException;
import java.net.InetAddress;
import java.net.NetworkInterface;
import java.net.SocketException;
import java.util.ArrayList;
import java.util.Enumeration;
import java.util.List;
import java.util.logging.Level;
import java.util.logging.Logger;

public class IpCache {

    private final static Logger LOG = Logger.getLogger(IpCache.class.getSimpleName());
    private static IpCache instance = null;
    private static final String LOCALHOST_V4 = "127.0.0.1";

    private IpCache() {
    }

    public static IpCache getInstance() {
        if (instance == null) {
            instance = new IpCache();
        }
        return instance;
    }

    public String getSubnetOfIp(String ip) {
        if (ip != null) {
            String[] parts = ip.split("\\.");
            if (parts.length == 4) {
                return parts[0] + "." + parts[1] + "." + parts[2];
            }
        }
        return ip;
    }

    public List<String> getIpsFromSubnet(String subnet) {
        String host;
        List<String> ips = new ArrayList<String>();
        if (subnet == null || LOCALHOST_V4.startsWith(subnet)) {
            return ips;
        }
        int timeout = 1000;
        for (int i = 1; i < 255; i++) {
            try {
                host = subnet + "." + i;
                if (InetAddress.getByName(host).isReachable(timeout)) {
                    LOG.log(Level.INFO, "subnet: {0} => {1}", new Object[]{subnet, host});
                    ips.add(host);
                }
            } catch (IOException ex) {
                LOG.log(Level.SEVERE, ex.getLocalizedMessage(), ex);
            }
        }
        return ips;
    }

    public List<String> getLocalIpv4s() {
        String ip;
        List<String> ips = new ArrayList<String>();
        // is always local :-)
        ips.add(LOCALHOST_V4);
        try {
            Enumeration<NetworkInterface> interfaces = NetworkInterface.getNetworkInterfaces();
            while (interfaces.hasMoreElements()) {
                NetworkInterface current = interfaces.nextElement();
                if (!current.isUp() || current.isLoopback() || current.isVirtual()) {
                    continue;
                }
                Enumeration<InetAddress> addresses = current.getInetAddresses();
                while (addresses.hasMoreElements()) {
                    InetAddress current_addr = addresses.nextElement();
                    ip = current_addr.getHostAddress().trim();
                    // Filter LoopBack and ipv6
                    if (ip == null
                            || ip.startsWith("fe")
                            || ip.length() > 15
                            || current_addr.isLoopbackAddress()) {
                        System.out.println("ip " + ip + " not valid!");
                        continue;
                    } else {
                        ips.add(ip);
                    }
                }
            }
            LOG.log(Level.FINER, "found {0} local ips", ips.size());
        } catch (SocketException ex) {
            LOG.log(Level.SEVERE, ex.getLocalizedMessage(), ex);
        }
        return ips;
    }
}
